# HRMS Responsive Design - Quick Reference Card

## 📱 Breakpoints Reference

```css
/* Mobile First Approach */
Base styles        → Mobile (320px+)
@media (min-width: 480px)   → Large Mobile
@media (min-width: 640px)   → Small Tablet
@media (min-width: 768px)   → Tablet
@media (min-width: 1024px)  → Laptop
@media (min-width: 1200px)  → Desktop
@media (min-width: 1440px)  → Large Desktop
```

## 🎯 Touch Target Sizes

```css
/* Minimum Sizes */
Buttons (Mobile)     → 48px × 48px
Small Buttons        → 44px × 44px
Input Fields         → 48px height
Icons (Clickable)    → 44px × 44px
Table Actions        → 44px × 44px
```

## 📐 Grid Patterns

### Stats Cards
```css
Desktop (1920px)  → 4 columns
Laptop (1366px)   → 3 columns
Tablet (1024px)   → 2 columns
Mobile (768px)    → 2 columns
Small Mobile      → 1 column
```

### Form Grids
```css
Desktop   → 2-3 columns
Tablet    → 2 columns
Mobile    → 1 column
```

### Quick Actions
```css
Desktop   → 3 columns
Tablet    → 2 columns
Mobile    → 1 column
```

## 🎨 Typography Scale

```css
/* Headings */
Desktop: h1 → 28px, h2 → 24px, h3 → 20px
Tablet:  h1 → 24px, h2 → 20px, h3 → 18px
Mobile:  h1 → 20px, h2 → 18px, h3 → 16px

/* Body Text */
All Devices → 14-15px (minimum 14px for readability)

/* Small Text */
All Devices → 12-13px
```

## 🔧 Common Responsive Patterns

### 1. Responsive Container
```css
.hrm-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

@media (max-width: 768px) {
    .hrm-container {
        padding: 0 16px;
    }
}
```

### 2. Responsive Grid
```css
.hrm-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

@media (max-width: 768px) {
    .hrm-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
}
```

### 3. Responsive Flexbox
```css
.hrm-flex {
    display: flex;
    gap: 16px;
}

@media (max-width: 768px) {
    .hrm-flex {
        flex-direction: column;
        gap: 12px;
    }
}
```

### 4. Responsive Table
```css
.hrm-table-wrapper {
    overflow-x: auto;
    -webkit-overflow-scrolling: touch;
}

.hrm-table {
    min-width: 800px;
}

@media (max-width: 768px) {
    .hrm-table {
        min-width: 600px;
    }
}
```

### 5. Responsive Modal
```css
.hrm-modal-content {
    max-width: 600px;
    width: 100%;
}

@media (max-width: 768px) {
    .hrm-modal-content {
        max-width: 100%;
        border-radius: 16px 16px 0 0;
    }
}
```

## 🎯 Utility Classes

### Visibility
```html
<!-- Hide on mobile -->
<div class="hide-mobile">Desktop only</div>

<!-- Show only on mobile -->
<div class="show-mobile">Mobile only</div>

<!-- Hide on tablet -->
<div class="hide-tablet">Not on tablet</div>
```

### Spacing
```css
.mb-mobile-16 { margin-bottom: 16px; } /* Mobile only */
.p-mobile-12 { padding: 12px; }        /* Mobile only */
```

## 📱 iOS Specific Fixes

### Prevent Zoom on Input Focus
```css
input, select, textarea {
    font-size: 16px; /* Minimum to prevent zoom */
}
```

### Safe Area Insets (Notched Devices)
```css
@supports (padding: max(0px)) {
    .container {
        padding-left: max(16px, env(safe-area-inset-left));
        padding-right: max(16px, env(safe-area-inset-right));
    }
}
```

### Disable Tap Highlight
```css
.button {
    -webkit-tap-highlight-color: transparent;
}
```

## 🎨 Common Mobile Patterns

### Stack on Mobile
```css
.stack-mobile {
    display: flex;
    gap: 16px;
}

@media (max-width: 768px) {
    .stack-mobile {
        flex-direction: column;
    }
}
```

### Full Width on Mobile
```css
.full-width-mobile {
    width: auto;
}

@media (max-width: 768px) {
    .full-width-mobile {
        width: 100%;
    }
}
```

### Hide Text, Show Icon
```css
.btn-responsive .text {
    display: inline;
}

@media (max-width: 480px) {
    .btn-responsive .text {
        display: none;
    }
}
```

## 🔍 Testing Quick Commands

### Chrome DevTools
```
Ctrl/Cmd + Shift + M  → Toggle Device Mode
Ctrl/Cmd + Shift + C  → Inspect Element
```

### Common Test Sizes
```
iPhone SE        → 375 × 667
iPhone 14 Pro    → 393 × 852
iPad             → 768 × 1024
iPad Pro         → 1024 × 1366
Desktop          → 1920 × 1080
```

## ⚡ Performance Tips

### Reduce Animations on Mobile
```css
@media (max-width: 768px) {
    * {
        animation-duration: 0.2s !important;
        transition-duration: 0.2s !important;
    }
}
```

### Optimize Images
```html
<img 
    src="image-mobile.jpg"
    srcset="image-mobile.jpg 480w,
            image-tablet.jpg 768w,
            image-desktop.jpg 1200w"
    sizes="(max-width: 768px) 100vw, 50vw"
    alt="Description"
>
```

## 🎯 Common Issues & Quick Fixes

### Issue: Horizontal Scroll
```css
/* Fix */
* {
    max-width: 100%;
    box-sizing: border-box;
}
```

### Issue: Text Too Small
```css
/* Fix */
body {
    font-size: 14px; /* Minimum */
}
```

### Issue: Buttons Too Small
```css
/* Fix */
@media (max-width: 768px) {
    .btn {
        min-height: 48px;
        min-width: 48px;
    }
}
```

### Issue: Modal Off Screen
```css
/* Fix */
@media (max-width: 768px) {
    .modal {
        max-width: 100%;
        margin: 0;
    }
}
```

## 📋 Pre-Launch Checklist

```
□ Test on iPhone (Safari)
□ Test on Android (Chrome)
□ Test on iPad
□ Test landscape orientation
□ Check touch targets (44px min)
□ Verify text readability (14px min)
□ Test forms without zoom
□ Check table scrolling
□ Verify modal behavior
□ Test navigation on mobile
□ Check button stacking
□ Verify image scaling
```

## 🎨 CSS Variables for Responsive

```css
:root {
    /* Spacing */
    --spacing-mobile: 12px;
    --spacing-tablet: 16px;
    --spacing-desktop: 20px;
    
    /* Font Sizes */
    --font-mobile: 14px;
    --font-tablet: 15px;
    --font-desktop: 16px;
    
    /* Touch Targets */
    --touch-target: 48px;
    --touch-target-small: 44px;
}
```

## 🚀 Quick Start Template

```css
/* Component Name */
.component {
    /* Base styles (mobile first) */
    padding: 16px;
    font-size: 14px;
}

/* Tablet */
@media (min-width: 768px) {
    .component {
        padding: 20px;
        font-size: 15px;
    }
}

/* Desktop */
@media (min-width: 1024px) {
    .component {
        padding: 24px;
        font-size: 16px;
    }
}
```

## 📞 Support Resources

- **Documentation**: `RESPONSIVE-COMPLETE.md`
- **Testing Guide**: `RESPONSIVE-TESTING-GUIDE.md`
- **Main CSS**: `assets/css/hrm-responsive-fixes.css`
- **Edge Cases**: `assets/css/hrm-responsive-edge-cases.css`

---

**Quick Tip**: Always test on real devices, not just DevTools!

**Last Updated**: 2026-01-23
**Version**: 1.0.0
